﻿using System;
using System.Linq;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrSchemas;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.Cache;

using HIPS.Client.Proxy;
using HIPS.CommonSchemas;
using HIPS.Web.Components.ServiceModel;

namespace HIPS.Web.Data.Hips.PcehrView
{

    /// <summary>
    /// Implements a repository for viewing the contents of a PCEHR.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
    /// </history>
    public class PcehrViewRepository : RepositoryBase<PCEHRProxy>, HIPS.Web.ModelInterface.PcehrView.IPcehrViewRepository
    {

        #region Constructors

        /// <summary>
        /// Initializes a new instance of the <see cref="PcehrViewRepository"/> class.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        ///   </history>
        public PcehrViewRepository() : this(new HIPS.Web.Components.Cache.NoCachingCacheProvider(), null)
        {
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="PcehrViewRepository"/> class.
        /// </summary>
        /// <param name="cacheProvider">Cache provider to be employed by the repository.</param>
        /// <param name="cacheKeyPrefix">Key prefix to be employed for caching.</param>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        ///   </history>
        public PcehrViewRepository(ICacheProvider cacheProvider, string cacheKeyPrefix = "") : base(cacheProvider, cacheKeyPrefix)
        {
            //Initialise client proxy ready for use.
            this.ClientProxy = new PCEHRProxy("PCEHREndPoint");
        }

        #endregion

        #region Methods

        /// <summary>
        /// Retrieves a list of active documents for a specified patient.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>DocumentListResponse containing the results of the operation.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        public ServiceResponse<DocumentListResponse<PatientIdentifierBase>> ListActiveDocuments(Mrn patientIdentifier, UserDetails operatingUser)
        {
            return this.CacheProvider.GetOrSetIf
                (
                    GetActiveDocumentsListFullCacheKey(patientIdentifier.HospitalCodeSystem, patientIdentifier.HospitalCode, patientIdentifier.Value),
                    () => this._ListActiveDocuments(patientIdentifier, operatingUser),
                    response => response.IsSuccessful
                );
        }

        /// <summary>
        /// Retrieves a specific document for a patient.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <param name="documentSpec">Identifier for the document.</param>
        /// <returns>DocumentResponse containing the results of the operation.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        public ServiceResponse<DocumentResponse> GetDocument(Mrn patientIdentifier, UserDetails operatingUser, DocumentRequest documentSpec)
        {
            // Perform a ListActiveDocuments call to establish the cache dependency parent.
            // Without the list being in cache the document won't make it into cache either.
            // However, the document would still be returned (just not cached).
            // This call should be quick if the list is cached.
            ListActiveDocuments(patientIdentifier, operatingUser);

            // Cache Dependency on current patient's Active Documents list
            return this.CacheProvider.GetOrSetIf
                (
                    this.GetMethodInvocationFullCacheKey(
                        System.Reflection.MethodInfo.GetCurrentMethod().Name,
                        new object[] { patientIdentifier.HospitalCodeSystem, patientIdentifier.HospitalCode, patientIdentifier.Value, documentSpec.RepositoryUniqueId, documentSpec.DocumentUniqueId }),
                    () => this._GetDocument(patientIdentifier, operatingUser, documentSpec),
                    response => response.IsSuccessful,
                    GetActiveDocumentsListFullCacheKey(patientIdentifier.HospitalCodeSystem, patientIdentifier.HospitalCode, patientIdentifier.Value)
                );
        }

        /// <summary>
        /// Retrieves a specific view for a patient.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <param name="viewRequest">Criteria for the view.</param>
        /// <returns>ViewResponse containing the requested view.</returns>
        public ServiceResponse<ViewResponse> GetView(Mrn patientIdentifier, UserDetails operatingUser, PrescriptionAndDispenseViewRequest viewRequest)
        {
            // Perform a ListActiveDocuments call to establish the cache dependency parent.
            // Without the list being in cache the document won't make it into cache either.
            // However, the document would still be returned (just not cached).
            // This call should be quick if the list is cached.
            ListActiveDocuments(patientIdentifier, operatingUser);

            // Cache Dependency on current patient's Active Documents list
            return CacheProvider.GetOrSetIf
                (
                    GetMethodInvocationFullCacheKey(
                        System.Reflection.MethodInfo.GetCurrentMethod().Name,
                        new object[] { patientIdentifier.HospitalCodeSystem, patientIdentifier.HospitalCode, patientIdentifier.Value, viewRequest.FromDate, viewRequest.ToDate }),
                    () => _GetView(patientIdentifier, operatingUser, viewRequest),
                    response => response.IsSuccessful,
                    GetActiveDocumentsListFullCacheKey(patientIdentifier.HospitalCodeSystem, patientIdentifier.HospitalCode, patientIdentifier.Value)
                );
        }

        /// <summary>
        /// Retrieves details of the patient's PCEHR status.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="dateOfBirth">Date of birth for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>DoesPcehrExistResponse containing the results of the operation.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="09 January 2014">Initial version.</change>
        /// </history>
        public ServiceResponse<DoesPcehrExistResponse> IsPcehrAdvertised(Mrn patientIdentifier, DateTime dateOfBirth, UserDetails operatingUser)
        {
            return this._IsPcehrAdvertised(patientIdentifier, dateOfBirth, operatingUser);
        }

        /// <summary>
        /// Attempts to gain access to a patient's PCEHR without using an access code.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>GainPcehrAccessResponse containing the results of the operation.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="07 January 2014">Initial version.</change>
        /// </history>
        public ServiceResponse<GainPcehrAccessResponse> GainAccessWithoutCode(Mrn patientIdentifier, UserDetails operatingUser)
        {
            ServiceResponse<GainPcehrAccessResponse> accessResponse = this._GenerateGainAccessServiceResponse(
                this.ClientProxy.GainAccessWithOutCode(patientIdentifier, operatingUser)
                );

            // If successfully executed a Gain Access then clear cache
            if (accessResponse.IsSuccessful)
            {
                string listCacheKey = GetActiveDocumentsListFullCacheKey(patientIdentifier.HospitalCodeSystem, patientIdentifier.HospitalCode, patientIdentifier.Value);
                CacheProvider.Set(listCacheKey, (object)null);
            }

            return accessResponse;
        }

        /// <summary>
        /// Attempts to gain access to a patient's PCEHR using an access code.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="accessCode">Access code provided by the user.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>GainPcehrAccessResponse containing the results of the operation.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="07 January 2014">Initial version.</change>
        /// </history>
        public ServiceResponse<GainPcehrAccessResponse> GainAccessWithCode(Mrn patientIdentifier, string accessCode, UserDetails operatingUser)
        {
            ServiceResponse<GainPcehrAccessResponse> accessResponse = this._GenerateGainAccessServiceResponse(
                this.ClientProxy.GainAccessWithCode(patientIdentifier, accessCode, operatingUser)
                );

            // If successfully executed a Gain Access then clear cache
            if (accessResponse.IsSuccessful)
            {
                string listCacheKey = GetActiveDocumentsListFullCacheKey(patientIdentifier.HospitalCodeSystem, patientIdentifier.HospitalCode, patientIdentifier.Value);
                CacheProvider.Set(listCacheKey, (object)null);
            }

            return accessResponse;
        }

        /// <summary>
        /// Attempts to gain access emergency to a patient's PCEHR.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>GainPcehrAccessResponse containing the results of the operation.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="07 January 2014">Initial version.</change>
        /// </history>
        public ServiceResponse<GainPcehrAccessResponse> GainAccessEmergency(Mrn patientIdentifier, UserDetails operatingUser)
        {
            ServiceResponse<GainPcehrAccessResponse> accessResponse = this._GenerateGainAccessServiceResponse(
                this.ClientProxy.GainAccessEmergency(patientIdentifier, operatingUser)
                );

            // If successfully executed a Gain Access then clear cache
            if (accessResponse.IsSuccessful)
            {
                string listCacheKey = GetActiveDocumentsListFullCacheKey(patientIdentifier.HospitalCodeSystem, patientIdentifier.HospitalCode, patientIdentifier.Value);
                CacheProvider.Set(listCacheKey, (object)null);
            }

            return accessResponse;
        }

        #region Private

        /// <summary>
        /// Retrieves a list of active documents for a specified patient.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>DocumentListResponse containing the results of the operation.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        private ServiceResponse<DocumentListResponse<PatientIdentifierBase>> _ListActiveDocuments(PatientIdentifierBase patientIdentifier, CommonSchemas.UserDetails operatingUser)
        {
            var result = this.ClientProxy.GetDocumentListActive(patientIdentifier, operatingUser);

            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            if (result != null)
            {
                // HIPS Status should be OK
                isSuccessful = result.HipsResponse.Status == HipsResponseIndicator.OK;

                // Add HIPS messages
                messages.AddRange(result.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                // Add a message if unable to get a response (shouldn't really happen)
                messages.Add("Unable to retrieve documents for the selected patient.", MessageLevel.Error);
            }

            return new ServiceResponse<DocumentListResponse<PatientIdentifierBase>>(result, isSuccessful, messages, DateTime.Now);
        }

        /// <summary>
        /// Retrieves a specific document for a patient.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <param name="documentSpec">Identifier for the document.</param>
        /// <returns>DocumentResponse containing the results of the operation.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="21 November 2013">Initial version.</change>
        /// </history>
        private ServiceResponse<DocumentResponse> _GetDocument(PatientIdentifierBase patientIdentifier, UserDetails operatingUser, DocumentRequest documentSpec)
        {
            var result = this.ClientProxy.RetrieveDocument(patientIdentifier, operatingUser, documentSpec);

            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            if (result != null)
            {
                // HIPS Status is irrelevant to the 'success' of the view retrieval.
                isSuccessful = result.Document != null;

                // Add an error message if the document cannot be retrieved
                if (result.Document == null)
                {
                    messages.Add("Unable to retrieve document to be displayed.", MessageLevel.Error);
                }

                // Add HIPS messages
                messages.AddRange(result.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                // Add a message if unable to get a response (shouldn't really happen)
                messages.Add("The requested document could not be retrieved.", MessageLevel.Error);
            }

            return new ServiceResponse<DocumentResponse>(result, isSuccessful, messages, DateTime.Now);
        }

        /// <summary>
        /// Retrieves a specific view for a patient.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <param name="viewRequest">Criteria for the view.</param>
        /// <returns>ViewResponse containing the requested view.</returns>
        public ServiceResponse<ViewResponse> _GetView(Mrn patientIdentifier, UserDetails operatingUser, PrescriptionAndDispenseViewRequest viewRequest)
        {
            var result = this.ClientProxy.GetView(patientIdentifier, operatingUser, viewRequest);

            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            if (result != null)
            {
                // HIPS Status is irrelevant to the 'success' of the view retrieval.
                isSuccessful = result.Document != null;

                // Add an error message if the document cannot be retrieved
                if (result.Document == null)
                {
                    messages.Add("Unable to retrieve view to be displayed.", MessageLevel.Error);
                }

                // Add HIPS messages
                messages.AddRange(result.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                // Add a message if unable to get a response (shouldn't really happen)
                messages.Add("The requested view could not be retrieved.", MessageLevel.Error);
            }
            
            return new ServiceResponse<ViewResponse>(result, isSuccessful, messages, DateTime.Now);
        }


        /// <summary>
        /// Retrieves details of the patient's PCEHR status.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="dateOfBirth">Date of birth for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>DoesPcehrExistResponse containing the results of the operation.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="09 January 2014">Initial version.</change>
        /// </history>
        private ServiceResponse<DoesPcehrExistResponse> _IsPcehrAdvertised(Mrn patientIdentifier, DateTime dateOfBirth, UserDetails operatingUser)
        {
            var result = this.ClientProxy.IsPcehrAdvertised(patientIdentifier, dateOfBirth, operatingUser);

            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            if (result != null)
            {
                // HIPS Status should be OK
                isSuccessful = result.HipsResponse.Status == HipsResponseIndicator.OK;

                // Add HIPS messages
                messages.AddRange(result.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                // Add a message if unable to get a response (shouldn't really happen)
                messages.Add("Unable to retrieve documents for the selected patient.", MessageLevel.Error);
            }

            return new ServiceResponse<DoesPcehrExistResponse>(result, isSuccessful, messages, DateTime.Now);
        }


        /// <summary>
        /// Generates the wrapped service response based on a gain access operation.
        /// </summary>
        /// <param name="gainAccessResponse">Result of the gain access operation.</param>
        /// <returns>GainPcehrAccessResponse containing the results of the operation.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="07 January 2014">Initial version.</change>
        /// </history>
        private ServiceResponse<GainPcehrAccessResponse> _GenerateGainAccessServiceResponse(GainPcehrAccessResponse gainAccessResponse)
        {
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            if (gainAccessResponse != null)
            {
                // HIPS Status should be OK
                isSuccessful = gainAccessResponse.HipsResponse.Status == HipsResponseIndicator.OK;

                // Add HIPS messages
                messages.AddRange(gainAccessResponse.HipsResponse.ToMessageListExpectOkStatus());
            }
            else
            {
                // Add a message if unable to get a response (shouldn't really happen)
                messages.Add("Unable to gain access for the specified patient.", MessageLevel.Error);
            }

            return new ServiceResponse<GainPcehrAccessResponse>(gainAccessResponse, isSuccessful, messages, DateTime.Now);
        }

        /// <summary>
        /// Gets the full cache key for a list of active documents.
        /// </summary>
        /// <param name="hospitalCodeSystem">The hospital code system.</param>
        /// <param name="hospitalCode">The hospital code.</param>
        /// <param name="patientIdentifier">The patient identifier.</param>
        /// <returns></returns>
        private string GetActiveDocumentsListFullCacheKey(string hospitalCodeSystem, string hospitalCode, string patientIdentifier)
        {
            // While the cache key won't match the method name, the name of this method will serve the same purpose
            return GetMethodInvocationFullCacheKey(
                        System.Reflection.MethodInfo.GetCurrentMethod().Name,
                        hospitalCodeSystem, hospitalCode, patientIdentifier);
        }

        #endregion

        #endregion

    }

}
